/*
 *  MoreSecurityAsync.c
 *  TestAuthTool
 *
 *  Created by Brian Wells on 12/4/05.
 *  Copyright 2005 Brian D. Wells. All rights reserved.
 *
 *  You may redistribute this software and/or modify it under
 *  the terms set forth in the accompanying LICENSE file.
 *
 */

#include "MoreSecurityAsync.h"

#include <unistd.h>
#include <sys/param.h>
#include <sys/socket.h>


/////////////////////////////////////////////////////////////////
#pragma mark ***** MoreSecurity Support Functions

/*
 * These are static functions that were copied over verbatim from the MoreSecurity.c file.
 * It would have been nice if they had been defined externally, but I decided to duplicate
 * them here instead of modifying MoreSecurity.c.
 *
 */

static void CheckWaitForDebuggerInterference(int *err, pid_t childPID, int *status)
    // The problem this is trying to solve is as follows.  When we use 
    // GDB to attach to our helper tool (in order to debug it), GDB 
    // temporarily changes the helper tool's parent process to be GDB. 
    // Thus, if we call waitpid while GDB is attached to the helper tool, 
    // waitpid returns ECHILD.
    // 
    // The solution is to set a breakpoint on this second call to waitpid. 
    // If you hit that breakpoint, you should either detach GDB from the 
    // helper tool or allow the helper tool to run to completion.  Either 
    // of these will restore the helper tool's parent process back to the 
    // application, which allows waitpid to succeed.
    //
    // I allow this code to run even in the non-debugging case because, 
    // other than the fprintf, it's pretty benign and it's certainly not 
    // on a performance critical path.
{
    if (*err == ECHILD) {
        #if MORE_DEBUG
            fprintf(stderr, "MoreSecurity: You need to have a breakpoint set here.\n");
        #endif
        *err = waitpid(childPID, status, 0);
        *err = MoreUNIXErrno(*err);
    }
}

static OSStatus CopyDictionaryFromDescriptor(int fdIn, CFDictionaryRef *dictResult)
	// Create a CFDictionary by reading the XML data from fdIn. 
	// It first reads the size of the XML data, then allocates a 
	// buffer for that data, then reads the data in, and finally 
	// unflattens the data into a CFDictionary.
	//
	// See also the companion routine, WriteDictionaryToDescriptor, below.
{
	OSStatus			err;
	CFIndex				dictSize;
	UInt8 *				dictBuffer;
	CFDataRef			dictData;
	CFPropertyListRef 	dict;

	assert(fdIn >= 0);
	assert( dictResult != NULL);
	assert(*dictResult == NULL);
	
	dictBuffer = NULL;
	dictData   = NULL;
	dict       = NULL;

	// Read the data size and allocate a buffer.
	
	err = EXXXToOSStatus( MoreUNIXRead(fdIn, &dictSize, sizeof(dictSize), NULL) );
	if (err == noErr) {
		if (dictSize == 0) {
			// malloc(0) may return NULL, so we specifically check for and error 
			// out in that case.
			err = paramErr;
		} else if (dictSize > (1 * 1024 * 1024)) {
			// Abitrary limit to prevent potentially hostile client overwhelming us with data.
			err = memFullErr;
		}
	}
	if (err == noErr) {
		dictBuffer = (UInt8 *) malloc( (size_t) dictSize);
		if (dictBuffer == NULL) {
			err = memFullErr;
		}
	}
	
	// Read the data and unflatten.
	
	if (err == noErr) {
		err = EXXXToOSStatus( MoreUNIXRead(fdIn, dictBuffer, (size_t) dictSize, NULL) );
	}
	if (err == noErr) {
		dictData = CFDataCreateWithBytesNoCopy(NULL, dictBuffer, dictSize, kCFAllocatorNull);
		err = CFQError(dictData);
	}
	if (err == noErr) {
		dict = CFPropertyListCreateFromXMLData(NULL, dictData, kCFPropertyListImmutable, NULL);
		err = CFQError(dict);
	}
	if ( (err == noErr) && (CFGetTypeID(dict) != CFDictionaryGetTypeID()) ) {
		err = paramErr;		// only CFDictionaries need apply
	}
	// CFShow(dict);
	
	// Clean up.
	
	if (err != noErr) {
		CFQRelease(dict);
		dict = NULL;
	}
	*dictResult = (CFDictionaryRef) dict;

	free(dictBuffer);
	CFQRelease(dictData);
	
	assert( (err == noErr) == (*dictResult != NULL) );
	
	return err;
}

static OSStatus WriteDictionaryToDescriptor(CFDictionaryRef dict, int fdOut)
	// Write a dictionary to a file descriptor by flattening 
	// it into XML.  Send the size of the XML before sending 
	// the data so that CopyDictionaryFromDescriptor knows 
	// how much to read.
{
	OSStatus			err;
	CFDataRef			dictData;
	CFIndex				dictSize;
	UInt8 *				dictBuffer;

	assert(dict != NULL);
	assert(fdOut >= 0);
	
	dictData   = NULL;
	dictBuffer = NULL;
	
	dictData = CFPropertyListCreateXMLData(NULL, dict);
	err = CFQError(dictData);
	
	// Allocate sizeof(size_t) extra bytes in the buffer so that we can 
	// prepend the dictSize.  This allows us to write the entire 
	// dict with one MoreUNIXWrite call, which definitely speeds 
	// things up, especially if this is was going over a real wire.
	// Of course, if I was to send this over a real wire, I would 
	// have to guarantee that dictSize was sent in network byte order (-:
	
	if (err == noErr) {
		dictSize = CFDataGetLength(dictData);
		dictBuffer = (UInt8 *) malloc( sizeof(size_t) + dictSize );
		if (dictBuffer == NULL) {
			err = memFullErr;
		}
	}
	
	if (err == noErr) {
		// Copy dictSize into the first size_t bytes of the buffer.
		
		*((size_t *) dictBuffer) = (size_t) dictSize;
		
		// Copy the data into the remaining bytes.
		//
		// Can't use CFDataGetBytePtr because there's no guarantee that 
		// it will succeed.  If it doesn't, we have to copy the bytes anyway,
		// so the allocation code has to exist.  Given that this isn't a 
		// performance critical path, I might as well minimise my code size by 
		// always running the allocation code.
		
		CFDataGetBytes(dictData, CFRangeMake(0, dictSize), dictBuffer + sizeof(size_t));
		
		err = EXXXToOSStatus( MoreUNIXWrite(fdOut, dictBuffer, sizeof(size_t) + dictSize, NULL) );
	}

	free(dictBuffer);
	CFQRelease(dictData);
		
	return err;
}

static int CopyDictionaryTranslatingDescriptors(int fdIn, CFDictionaryRef *response)
	// Reads a dictionary and its associated descriptors from fdIn, 
	// putting the dictionary (modified to include the translated 
	// descriptor numbers) in *response.
{
	int 				err;
	int 				junk;
	CFDictionaryRef		receivedResponse;
	CFArrayRef 			incomingDescs;
	
	assert(fdIn >= 0);
	assert( response != NULL);
	assert(*response == NULL);
	
	receivedResponse = NULL;
	
	// Read the dictionary.
	
	err = OSStatusToEXXX( CopyDictionaryFromDescriptor(fdIn, &receivedResponse) );
	
	// Now read the descriptors, if any.
	
	if (err == 0) {
		incomingDescs = (CFArrayRef) CFDictionaryGetValue(receivedResponse, kMoreSecFileDescriptorsKey);
		if (incomingDescs == NULL) {
			// No descriptors.  Not much to do.  Just use receivedResponse as 
			// the response.
			
			*response = receivedResponse;
			receivedResponse = NULL;
		} else {
			CFMutableArrayRef 		translatedDescs;
			CFMutableDictionaryRef	mutableResponse;
			CFIndex					descCount;
			CFIndex					descIndex;
			
			// We have descriptors, so there's lots of stuff to do.  Have to 
			// receive each of the descriptors assemble them into the 
			// translatedDesc array, then create a mutable dictionary based 
			// on response (mutableResponse) and replace the 
			// kMoreSecFileDescriptorsKey with translatedDesc.
			
			translatedDescs  = NULL;
			mutableResponse  = NULL;

			// Start by checking incomingDescs.
					
			if ( CFGetTypeID(incomingDescs) != CFArrayGetTypeID() ) {
				err = kCFQDataErr;
			}
			
			// Create our output data.
			
			if (err == 0) {
				err = CFQArrayCreateMutable(&translatedDescs);
			}
			if (err == 0) {
				mutableResponse = CFDictionaryCreateMutableCopy(NULL, 0, receivedResponse);
				if (mutableResponse == NULL) {
					err = OSStatusToEXXX( coreFoundationUnknownErr );
				}
			}

			// Now read each incoming descriptor, appending the results 
			// to translatedDescs as we go.  By keeping our working results 
			// in translatedDescs, we make sure that we can clean up if 
			// we fail.
			
			if (err == 0) {
				descCount = CFArrayGetCount(incomingDescs);
				
				// We don't actually depend on the descriptor values in the 
				// response (that is, the elements of incomingDescs), because 
				// they only make sense it the context of the sending process. 
				// All we really care about is the number of elements, which 
				// tells us how many times to go through this loop.  However, 
				// just to be paranoid, in the debug build I check that the 
				// incoming array is well formed.

				#if MORE_DEBUG
					for (descIndex = 0; descIndex < descCount; descIndex++) {
						int 		thisDesc;
						CFNumberRef thisDescNum;
						
						thisDescNum = (CFNumberRef) CFArrayGetValueAtIndex(incomingDescs, descIndex);
						assert(thisDescNum != NULL);
						assert(CFGetTypeID(thisDescNum) == CFNumberGetTypeID());
						assert(CFNumberGetValue(thisDescNum, kCFNumberIntType, &thisDesc));
						assert(thisDesc >= 0);
					}
				#endif
				
				// Here's the real work.  For descCount times, read a descriptor 
				// from fdIn, wrap it in a CFNumber, and append it to translatedDescs. 
				// Note that we have to be very careful not to leak a descriptor 
				// if we get an error here.
				
				for (descIndex = 0; descIndex < descCount; descIndex++) {
					int 		thisDesc;
					CFNumberRef thisDescNum;
					
					thisDesc = -1;
					thisDescNum = NULL;
					
					err = MoreUNIXReadDescriptor(fdIn, &thisDesc);
					if (err == 0) {
						thisDescNum = CFNumberCreate(NULL, kCFNumberIntType, &thisDesc);
						if (thisDescNum == NULL) {
							err = OSStatusToEXXX( coreFoundationUnknownErr );
						}
					}
					if (err == 0) {
						CFArrayAppendValue(translatedDescs, thisDescNum);
						// The descriptor is now stashed in translatedDescs, 
						// so this iteration of the loop is no longer responsible 
						// for closing it.
						thisDesc = -1;		
					}
					
					CFQRelease(thisDescNum);
					if (thisDesc != -1) {
						junk = close(thisDesc);
						assert(junk == 0);
					}
					
					if (err != 0) {
						break;
					}
				}
			}

			// Clean up and establish output parameters.
			
			if (err == 0) {
				CFDictionarySetValue(mutableResponse, kMoreSecFileDescriptorsKey, translatedDescs);
				*response = mutableResponse;
			} else {
				MoreSecCloseDescriptorArray(translatedDescs);
				CFQRelease(mutableResponse);
			}
			CFQRelease(translatedDescs);
		}
	}
	
	CFQRelease(receivedResponse);
	
	assert( (err == 0) == (*response != NULL) );
	
	return err;
}

static int WriteDictionaryAndDescriptors(CFDictionaryRef response, int fdOut)
	// Writes a dictionary and its associated descriptors to fdOut.
{
	int 			err;
	CFArrayRef 		descArray;
	CFIndex			descCount;
	CFIndex			descIndex;
	
	// Write the dictionary.
	
	err = OSStatusToEXXX( WriteDictionaryToDescriptor(response, fdOut) );
	
	// Process any descriptors.  The descriptors are indicated by 
	// a special key in the dictionary.  If that key is present, 
	// it's a CFArray of CFNumbers that present the descriptors to be 
	// passed.
	
	if (err == 0) {
		descArray = (CFArrayRef) CFDictionaryGetValue(response, kMoreSecFileDescriptorsKey);
		
		// We only do the following if the special key is present.
		
		if (descArray != NULL) {
		
			// If it's not an array, that's bad.
			
			if ( CFGetTypeID(descArray) != CFArrayGetTypeID() ) {
				err = EINVAL;
			}
			
			// Loop over the array, getting each descriptor and writing it.
			
			if (err == 0) {
				descCount = CFArrayGetCount(descArray);
				
				for (descIndex = 0; descIndex < descCount; descIndex++) {
					CFNumberRef thisDescNum;
					int 		thisDesc;
					
					thisDescNum = (CFNumberRef) CFArrayGetValueAtIndex(descArray, descIndex);
					if (   (thisDescNum == NULL) 
						|| (CFGetTypeID(thisDescNum) != CFNumberGetTypeID()) 
						|| ! CFNumberGetValue(thisDescNum, kCFNumberIntType, &thisDesc) ) {
						err = EINVAL;
					}
					if (err == 0) {
						err = MoreUNIXWriteDescriptor(fdOut, thisDesc);
					}

					if (err != 0) {
						break;
					}
				}
			}
		}
	}

	return err;
}

/////////////////////////////////////////////////////////////////
#pragma mark ***** Asynchronous Usage Functions

extern OSStatus MoreSecAsyncExecuteRequestInHelperTool(CFURLRef helperTool, AuthorizationRef auth, 
											 	  CFDictionaryRef request, int *fdPipe, int *childPID)
{
	OSStatus 					err;
	int 						junk;
	char						toolPath[MAXPATHLEN];
    AuthorizationExternalForm 	extAuth;
    int							fdChild;
    int							fdParent;
    int							processPID;
	CFMutableDictionaryRef tempDict;
	CFNumberRef asyncNum;
	int asyncValue;

	assert(helperTool != NULL);
	assert(auth       != NULL);
	assert(request    != NULL);
	assert( fdPipe	  != NULL);
	assert( childPID  != NULL);
	
	processPID = -1;
	fdChild  = -1;
	fdParent = -1;
	
	*fdPipe = -1;
	*childPID = -1;
	
	// get tool path and authorization external form
		
	err = CFQErrorBoolean( CFURLGetFileSystemRepresentation(helperTool, true, (UInt8 *)toolPath, sizeof(toolPath)) );
	if (err == noErr) {
		err = AuthorizationMakeExternalForm(auth, &extAuth);
	}
	
	// Create a pair of anonymous UNIX domain sockets
	
	if (err == noErr) {
		int comm[2];
		
		err = socketpair(AF_UNIX, SOCK_STREAM, 0, comm);
		err = EXXXToOSStatus( MoreUNIXErrno(err) );
		
		if (err == noErr) {
			fdChild  = comm[0];
			fdParent = comm[1];
		}
	}
	
	// fork child process
		
	if (err == noErr) {
		processPID = vfork();

		if (processPID == 0) {						// Child
			err = dup2(fdChild, STDIN_FILENO);
			err = EXXXToOSStatus( MoreUNIXErrno(err) );
			
			if (err == noErr) {
				err = dup2(fdChild, STDOUT_FILENO);
				err = EXXXToOSStatus( MoreUNIXErrno(err) );
			}

			if (err == noErr) {
				junk = close(fdChild);
				assert(junk == 0);
				junk = close(fdParent);
				assert(junk == 0);

			    err = execl(toolPath, toolPath, NULL);
			    err = EXXXToOSStatus( MoreUNIXErrno(err) );
			}
		    assert(err != noErr);	// otherwise we wouldn't be here
		    _exit(MoreSecErrorToHelperToolResult(OSStatusToEXXX(err)));
			assert(false);			// unreached
		} else if (processPID == -1) {				// Parent - Error
			err = EXXXToOSStatus( MoreUNIXErrno(processPID) );
		} else {
			assert(processPID > 0);					// Parent
		}
	}
	
	// close fdChild
	
	if (fdChild != -1) {
		junk = close(fdChild);
		assert(junk == 0);
		fdChild = -1;
	}
	
	// write external authorization ref
	
	if (err == noErr) {	
		err = EXXXToOSStatus( MoreUNIXWrite(fdParent, &extAuth, sizeof(extAuth), NULL) );
	}
	
	// add key to request dict

	if (err == noErr) {
		tempDict = CFDictionaryCreateMutableCopy(NULL, 0, request);
		err = OSStatusToEXXX( CFQError(tempDict) );

		if (err == 0) {
			asyncValue = 1;
			asyncNum = CFNumberCreate(NULL, kCFNumberIntType, &asyncValue);
			err = OSStatusToEXXX( CFQError(asyncNum) );
			if (err == 0) {
				CFDictionarySetValue(tempDict, kMoreSecAsyncIdentifierKey, asyncNum);
			}		
			CFQRelease(asyncNum);
		}
	}

	// write request dict to tool
	
	if (err == noErr) {
		err = WriteDictionaryToDescriptor(tempDict, fdParent);
	}

	// Close write side of pipe - now for reading only

	if (err == noErr) {
		err = shutdown(fdParent, 1);
		err = EXXXToOSStatus( MoreUNIXErrno(err) );
	}

	// return childPID and fdPipe

	if (err == noErr) {
		*childPID = processPID;
		*fdPipe = fdParent;
	}

	// close pipe to child if error - child will die once it realizes the pipe is closed (maybe)
	
	if (err != noErr && fdParent != -1) {
		junk = close(fdParent);
		junk = MoreUNIXErrno(junk);
		assert(junk == 0);
	}

	return err;
}

extern OSStatus MoreSecAsyncGetHelperToolResponse(int fdPipe, int childPID, CFDictionaryRef *response, Boolean *isFinalResponse)
{
	OSStatus err;
	short revents;
	int junk;
	int err2;
	int status;

	assert(fdPipe >= 0);
	assert(childPID != 0);
	assert( response  != NULL);
	assert(*response  == NULL);
	assert(isFinalResponse != NULL);
	assert(*isFinalResponse == false);

	// init response

	*response = NULL;
	*isFinalResponse = false;
	revents = 0;

	// poll for response

	err = MoreUNIXPoll(fdPipe, POLLIN, &revents, 0);	// 0 timeout = no blocking
	if (err == noErr) {
		if (revents & POLLIN) {
			// response waiting
			err = EXXXToOSStatus( CopyDictionaryTranslatingDescriptors(fdPipe, response) );
			// get final response flag
			if ((err == noErr) && (*response != NULL)) {
				*isFinalResponse = !(CFDictionaryContainsKey(*response, kMoreSecAsyncIdentifierKey));
			}
		}
		if (revents & POLLNVAL) {
			// bad descriptor
			err = EBADF;
		}
	}
	
	if (*isFinalResponse || err != noErr) {
		// close pipe
		junk = close(fdPipe);
		junk = MoreUNIXErrno(junk);
		assert(junk == 0);

		// wait for child process to end
	    err2 = waitpid(childPID, &status, 0);
        err2 = MoreUNIXErrno(err2);
        
        CheckWaitForDebuggerInterference(&err2, childPID, &status);

	    if (err == noErr) {
	    	err = EXXXToOSStatus( err2 );
	    }

		if ( (err == noErr) || (err == EPIPE) ) {
			if ( ! WIFEXITED(status) ) {
				err = kMoreSecResultInternalErrorErr;
			} else {
				err = EXXXToOSStatus( MoreSecHelperToolResultToError(WEXITSTATUS(status)) );
			}
		}

	}

	if (err != noErr) {
		if (*response != NULL) {
			MoreSecCloseDescriptorArray((CFArrayRef) CFDictionaryGetValue(*response, kMoreSecFileDescriptorsKey));
		}
		CFQRelease(*response);
		*response = NULL;
		*isFinalResponse = true;
	}
	
	return err;
}

/*
extern OSStatus MoreSecAsyncAbortHelperToolRequest(int fdPipe, int childPID)
{
	OSStatus err;
	int junk;

	// close pipe

	if (fdPipe != -1) {
		junk = close(fdPipe);
		junk = MoreUNIXErrno(junk);
		assert(junk == 0);
	}

	// terminate child process
	
	if (childPID != -1) {
		// kill(childPID, SIGKILL) does not work because root owns the tool
		// Tool might die on its own once it realizes the pipe is closed, but that is not ideal
		// TO DO: Find a way to tell the tool it needs to terminate

		err = kill(childPID, SIGTERM);
		err = MoreUNIXErrno(err);
		if (err == ESRCH) {
			// not found - ignore
			err = noErr;
		}


	}

	return err;
}
*/

extern OSStatus MoreSecAsyncHelperToolWriteResponse(CFDictionaryRef dict, int fdOut)
{
	OSStatus err;
	CFMutableDictionaryRef tempDict;
	CFNumberRef errNum;
	CFNumberRef asyncNum;
	int asyncValue;

	assert(dict != NULL);
	assert(fdOut >= 0);

	// create mutable copy of dictionary
	
	tempDict = CFDictionaryCreateMutableCopy(NULL, 0, dict);
	err = OSStatusToEXXX( CFQError(tempDict) );

	// add kMoreSecErrorNumberKey if missing

	if ((err == 0) && (! CFDictionaryContainsKey(tempDict, kMoreSecErrorNumberKey))) {
		errNum = CFNumberCreate(NULL, kCFNumberSInt32Type, &err);
		err = OSStatusToEXXX( CFQError(errNum) );
		if (err == 0) {
			CFDictionarySetValue(tempDict, kMoreSecErrorNumberKey, errNum);
		}		
		CFQRelease(errNum);
	}

	// add kMoreSecAsyncIdentifierKey

	if (err == 0) {
		asyncValue = 1;
		asyncNum = CFNumberCreate(NULL, kCFNumberIntType, &asyncValue);
		err = OSStatusToEXXX( CFQError(asyncNum) );
		if (err == 0) {
			CFDictionarySetValue(tempDict, kMoreSecAsyncIdentifierKey, asyncNum);
		}		
		CFQRelease(asyncNum);
	}

	// write response
		
	if (err == 0) {
		err = WriteDictionaryAndDescriptors(tempDict, fdOut);
	}
	
	// close descriptors (if any)
	
	if (tempDict != NULL) {
		MoreSecCloseDescriptorArray((CFArrayRef) CFDictionaryGetValue(dict, kMoreSecFileDescriptorsKey));
	}
	
	// release mutable copy
	
	CFQRelease(tempDict);

	return err;
}


/////////////////////////////////////////////////////////////////
#pragma mark ***** UNIX Support Functions

extern int MoreUNIXPoll(int fd, short events, short *revents, int timeout)
	// See comment in header.
{
	int err;
	struct pollfd pfd;

	assert(fd >= 0);
	assert(revents != NULL);

	err = 0;
	
	pfd.fd = fd;
	pfd.events = events;
	pfd.revents = 0;
	*revents = 0;
	
	while (err == 0) {
		err = poll(&pfd, 1, timeout);
		err = MoreUNIXErrno(err);
		if (err == -1) {
			err = errno;
			if (err == EINTR) {
				err = 0;	// ignore interrupt - loop again
			}
		} else {
			// done polling...
			assert(!(pfd.revents & POLLERR));
			assert(!(pfd.revents & POLLNVAL));
			// return matching events 
			*revents = pfd.revents;
			break;
		}
	}
	
	return err;
}

