//
//  TestAuthToolController.m
//  TestAuthTool
//
//  Created by Brian Wells on 11/21/05.
//  Copyright 2005 Brian D. Wells. All rights reserved.
//
//  You may redistribute this software and/or modify it under
//  the terms set forth in the accompanying LICENSE file.
//

#import "TestAuthToolController.h"

@implementation TestAuthToolController

- (id)init
{
	if (self = [super init]) {
		isRunningTest = NO;
	}
	return self;
}

- (IBAction)activateTool:(id)sender
{
	OSStatus err;
	NSDictionary *request;
	NSDictionary *response = NULL;

    err = AuthorizationCreate(NULL, kAuthorizationEmptyEnvironment, kAuthorizationFlagDefaults, &auth);

	if (err == noErr) {
		err = MoreSecCopyHelperToolURLAndCheckBundled(
													  CFBundleGetMainBundle(), 
													  (CFStringRef) @"TestAuthToolTemplate", 
													  kApplicationSupportFolderType, 
													  (CFStringRef) @"TestAuthTool", 
													  (CFStringRef) @"TestAuthToolTool", 
													  (CFURLRef *) &tool
													  );
	
		// If the home directory is on an volume that doesn't support 
		// setuid root helper tools, ask the user whether they want to use 
		// a temporary tool.
			
		if (err == kMoreSecFolderInappropriateErr) {
			
			int itemHit = NSRunAlertPanel(
				@"Your home directory is on a volume that does not support privileged helper tools. \nWould you like to use a temporary copy of the tool?",
				@"The temporary tool will be deleted periodically. In the Finder's Get Info window, uncheck the 'Ignore ownership' on the disk containing your home directory. Alternatively, ask your system administrator to install the tool for you.",
				@"OK", @"Cancel",nil,nil);
			
				
			if (itemHit == NSAlertErrorReturn) {
				err = coreFoundationUnknownErr;
			}
				
			if ( (err == noErr) && (itemHit == NSAlertAlternateReturn) ) {
				err = userCanceledErr;
			}
				
			if (err == noErr) {
				err = MoreSecCopyHelperToolURLAndCheckBundled(
															  CFBundleGetMainBundle(), 
															  (CFStringRef) @"TestAuthToolTemplate", 
															  kTemporaryFolderType, 
															  (CFStringRef) @"TestAuthTool", 
															  (CFStringRef) @"TestAuthToolTool", 
															  (CFURLRef *) &tool
															  );
			}
		}
	}

	if (err == noErr) {
		request = [NSDictionary dictionaryWithObjectsAndKeys:(NSString *)kTestAuthToolActivateCommand, (NSString *) kTestAuthToolCommandNameKey, nil];
		err = CFQError(request);
	
		if (err == noErr) {
			err = MoreSecExecuteRequestInHelperTool((CFURLRef) tool, auth, (CFDictionaryRef) request, (CFDictionaryRef *) &response);
		
			if (err == noErr) {
				err = MoreSecGetErrorFromResponse((CFDictionaryRef) response);
			}
		}
	}

	if (err == noErr) {
		// tool registered... update UI
		
		[activateButton setEnabled:NO];
		[checkButton setEnabled:YES];
		
		[testButton setEnabled:YES];
		[testInputField setEnabled:YES];
		[testAsnycButton setEnabled:YES];
	}
		
	// notify user of error!
	if (err != noErr && err != userCanceledErr) {
		NSRunCriticalAlertPanel(@"Error",
			[NSString stringWithFormat:@"Unable to authorize and or install helper tool (%d)", err],
			@"Continue",nil,nil
			);
	}

}

- (IBAction)checkUIDs:(id)sender
{
	OSStatus err;
	NSDictionary *request;
	NSDictionary *response = NULL;
	NSNumber *result;
	
	request = [NSDictionary dictionaryWithObjectsAndKeys:(NSString *)kTestAuthToolCheckUIDsCommand, (NSString *) kTestAuthToolCommandNameKey, nil];
	err = CFQError(request);
	
	if (err == noErr) {
		err = MoreSecExecuteRequestInHelperTool((CFURLRef) tool, auth, (CFDictionaryRef) request, (CFDictionaryRef *) &response);
		
		if (err == noErr) {
			err = MoreSecGetErrorFromResponse((CFDictionaryRef) response);
		}
	}

	if (err == noErr) {
		result = (NSNumber *)[response objectForKey:(NSString *)kTestAuthToolCheckUIDsResponseEUID];
		[euidField setStringValue:(result == nil ? @"?" : [result stringValue])];
		result = (NSNumber *)[response objectForKey:(NSString *)kTestAuthToolCheckUIDsResponseRUID];
		[ruidField setStringValue:(result == nil ? @"?" : [result stringValue])];
		result = (NSNumber *)[response objectForKey:(NSString *)kTestAuthToolCheckUIDsResponseSUID];
		[suidField setStringValue:(result == nil ? @"?" : [result stringValue])];
	}
	
	// notify user of error!
	if (err != noErr) {
		NSRunCriticalAlertPanel(@"Error",
			[NSString stringWithFormat:@"Unable to check UIDs (%d)", err],
			@"Continue",nil,nil
			);
	}

}

- (IBAction)runTest:(id)sender
{
	OSStatus err;
	NSDictionary *request;
	NSDictionary *response = NULL;
	NSString *result;
	int junk;
	
	if (isRunningTest) {
		// abort test
		[timer invalidate];
		[timer release];

		// close pipe
		if (fdPipe != -1) {
			junk = close(fdPipe);
			junk = MoreUNIXErrno(junk);
			assert(junk == 0);
		}

		// kill process by asking the tool
		if (childPID != -1) {
			request = [NSDictionary dictionaryWithObjectsAndKeys:
				(NSString *)kTestAuthToolAbortCommand, (NSString *) kTestAuthToolCommandNameKey,
				[NSNumber numberWithInt:childPID], kTestAuthToolAbortRequestProcessKey,
				nil];
			err = CFQError(request);
			if (err == noErr) {
				err = MoreSecExecuteRequestInHelperTool((CFURLRef) tool, auth, (CFDictionaryRef) request, (CFDictionaryRef *) &response);
				if (err == noErr) {
					err = MoreSecGetErrorFromResponse((CFDictionaryRef) response);
				}
			}
		}
		
		// update GUI
		[testButton setTitle:@"Test"];
		[testAsnycButton setEnabled:YES];
		[testInputField setEnabled:YES];
		isRunningTest = NO;
		return;
	}
	
	// clear results
	[testResultView setString:@""];

	// prepare request
	request = [NSDictionary dictionaryWithObjectsAndKeys:
		(NSString *)kTestAuthToolTestCommand, (NSString *) kTestAuthToolCommandNameKey,
		[testInputField stringValue], kTestAuthToolTestRequestStringKey,
		nil];
	err = CFQError(request);

	if ([testAsnycButton state] == NSOnState) {
		// asynchronous
		if (err == noErr) {
			err = MoreSecAsyncExecuteRequestInHelperTool((CFURLRef) tool, auth, (CFDictionaryRef) request, &fdPipe, &childPID);
		}
		
		if (err == noErr) {
			// start NSTimer with interval defined by kTestAuthToolPollInterval (TestAuthToolCommon.h)
			timer = [[NSTimer scheduledTimerWithTimeInterval:kTestAuthToolPollInterval
													  target:self
													selector:@selector(pollTest:)
													userInfo:nil
													 repeats:YES] retain];
			// update GUI
			[testButton setTitle:@"Stop"];
			[testAsnycButton setEnabled:NO];
			[testInputField setEnabled:NO];
			isRunningTest = YES;

		}
		
	} else {
		// synchronous
		
		if (err == noErr) {
			err = MoreSecExecuteRequestInHelperTool((CFURLRef) tool, auth, (CFDictionaryRef) request, (CFDictionaryRef *) &response);
		
			if (err == noErr) {
				err = MoreSecGetErrorFromResponse((CFDictionaryRef) response);
			}
		}

		// get response (only one)
		if (err == noErr) {
			result = (NSString *)[response objectForKey:(NSString *)kTestAuthToolTestResponseString];
			[testResultView setString:(result == nil ? @"" : result)];
			[self performSelector:@selector(scrollToVisible:) withObject:nil afterDelay:0.0];
		}
	}
	
	// notify user of error!
	if (err != noErr) {
		NSRunCriticalAlertPanel(@"Error",
			[NSString stringWithFormat:@"Unable to run test (%d)", err],
			@"Continue",nil,nil
			);
	}

}

- (void)pollTest:(NSTimer *)aTimer
{
	OSStatus err;
	NSDictionary *response;
	Boolean isFinal = false;
	NSString *result;
	int maxResponses = kTestAuthToolPollMaxResponses;	// defined in TestAuthToolCommon.h
	int numResponses = 0;
	
	/*
	 * This is where the parent process has the opportunity to poll the tool for any responses.
	 * 
	 * The responsiveness of the GUI can be tuned with the kTestAuthToolPollInterval and 
	 * kTestAuthToolPollMaxResponses values defined in TestAuthToolCommon.h.
	 *
	 * Care must be taken not to either stall the GUI or have responses pile up in the pipe.
	 *
	 */

	do {
		// MoreSecAsyncGetHelperToolResponse expects response to be NULL
		response = NULL;
		
		// get waiting response (if any)
		err = MoreSecAsyncGetHelperToolResponse(fdPipe, childPID, (CFDictionaryRef *)&response, &isFinal);
		if (err == noErr && response != NULL) {
			err = MoreSecGetErrorFromResponse((CFDictionaryRef) response);
		}
	
		// display error if need be (response will be NULL and isFinal will be true)
		if (err != noErr) {
			NSRunCriticalAlertPanel(@"Error",
				[NSString stringWithFormat:@"Problem fetching response (%d)", err],
				@"Continue",nil,nil
				);
		}

		// display successful responses
		if (response != NULL) {
			result = (NSString *)[response objectForKey:(NSString *)kTestAuthToolTestResponseString];
			if (result != nil) {
				[[testResultView textStorage] appendAttributedString: [[[NSAttributedString alloc] 
					initWithString:result] autorelease]];
				[self performSelector:@selector(scrollToVisible:) withObject:nil afterDelay:0.0];
			}
		}
	
		// reset GUI if final response or an error
		if (isFinal) {
			[timer invalidate];
			[timer release];
			[testButton setTitle:@"Test"];
			[testAsnycButton setEnabled:YES];
			[testInputField setEnabled:YES];
			isRunningTest = NO;
		}
		
		// check to see if we handled maximum number of responses
		if (maxResponses > 0) {
			numResponses++;
			if (numResponses == maxResponses) {
				response == NULL;
			}
		}
		
	} while (response != NULL && isFinal != true);

}

- (void)scrollToVisible:(id)Ignore
{
	[testResultView scrollRangeToVisible:NSMakeRange([[testResultView string] length],0)];
}

- (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication *)theApplication
{
	return YES;
}

- (void)applicationWillTerminate:(NSNotification *)aNotification
{
	int junk;
	
	if (isRunningTest) {
		// this should cause the test to abort and clean up
		[self runTest:self];
	}

	if (auth != NULL) {	
		junk = AuthorizationFree(auth, kAuthorizationFlagDestroyRights);
		assert(junk == noErr);
	}
}

@end
