/*
 *  TestAuthToolTool.c
 *  TestAuthTool
 *
 *  Created by Brian Wells on 11/21/05.
 *  Copyright 2005 Brian D. Wells. All rights reserved.
 *
 *  You may redistribute this software and/or modify it under
 *  the terms set forth in the accompanying LICENSE file.
 *
 */

#include "TestAuthToolTool.h"


static char* Copy_CFStringRefToCString(const CFStringRef stringRef) {

    char *result = NULL;
    if (stringRef != NULL) {
        CFIndex length = sizeof(UniChar) * CFStringGetLength(stringRef) + 1;
        result = (char*) NewPtrClear(length);
        if (!CFStringGetCString(stringRef,result,length,kCFStringEncodingASCII)) {
            if (!CFStringGetCString(stringRef,result,length,kCFStringEncodingUTF8)) {
                DisposePtr(result);
                result = NULL;
            }
        }
    }
    return result;
}

static OSStatus DoTestCommand(CFDictionaryRef *result, CFStringRef reqStr, Boolean isAsync)
{
	OSStatus err;
	uid_t realUser;
	const char *cmdStr;
	FILE *fp;
	const UInt8 *line;
	CFMutableStringRef resStr;
	CFStringRef tempStr;
	CFIndex len = 0;
	CFDictionaryRef tempResult;

	CFStringRef keys[1];
	CFStringRef values[1];
	
	err = noErr;
	
	// NOTE: Initially this ran the requested command-line as root. However, it is fairly
	// risky to execute an arbitrary command as root since a typo could have unintended
	// consequences. So now the relavent lines have been commented out...
/*	
	// elevate privs
	err = MoreSecSetPrivilegedEUID();
	if (err == noErr) {
		// setting EUID to root is not enough - also set RUID
		realUser = getuid();
		setuid(0);
*/
		cmdStr = Copy_CFStringRefToCString(reqStr);
		fp = popen(cmdStr,"r");
		err = CFQError(fp);

		resStr = CFStringCreateMutable(NULL,0);
		err = CFQError(resStr);

		// read output of program
		while (err == noErr && (line = (UInt8 *)fgetln(fp,(size_t *)&len)) != NULL) {
			tempStr = CFStringCreateWithBytes(NULL,line,len,kCFStringEncodingASCII,false);
			err = CFQError(tempStr);
			if (isAsync) {
				// send response now
				keys[0] = kTestAuthToolTestResponseString;
				values[0] = tempStr;
				tempResult = CFDictionaryCreate(NULL, (const void **) &keys, (const void **) &values, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
				err = CFQError(tempResult);
				if (err == noErr) {
					err = MoreSecAsyncHelperToolWriteResponse(tempResult, STDOUT_FILENO);
				}
				CFQRelease(tempResult);
			} else {
				// append to CFMutableString
				CFStringAppend(resStr,tempStr);
			}
			CFQRelease(tempStr);
		}
		
		pclose(fp);
		
		DisposePtr((Ptr)cmdStr);

/*
		// set RUID back to real user
		setuid(realUser);

		// back down privs
		MoreSecTemporarilySetNonPrivilegedEUID();
	}
*/
	
	if (err == noErr) {
		keys[0] = kTestAuthToolTestResponseString;
		values[0] = (CFStringRef)resStr;

		err = noErr;
		if (values[0] == NULL) {
			err = coreFoundationUnknownErr;
		}
		if (err == noErr) {
			*result = CFDictionaryCreate(NULL, (const void **) &keys, (const void **) &values, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
			err = CFQError(*result);
		}
	
		CFQRelease(values[0]);
	}

	assert( (err == noErr) == (*result != NULL) );
		
	return err;
}

static OSStatus DoCheckUIDsCommand(CFDictionaryRef *result)
{
	OSStatus err;
	uid_t euid;
	uid_t ruid;
	uid_t suid;
	int junk;
	CFStringRef keys[3];
	CFNumberRef values[3];
	long long tmp;

	assert( result != NULL);
	assert(*result == NULL);

	// get response

	euid = geteuid();
	ruid = getuid();

	// There's no direct accessor for the SUID, so I try to 
	// set the EUID to 0.  If that works, our SUID must be 0 (-:
	
	if ( (euid != 0) && (MoreSecSetPrivilegedEUID() == 0) ) {
		suid = 0;
		junk = MoreSecTemporarilySetNonPrivilegedEUID();
		assert(junk == 0);
	} else {
		suid = ruid;
	}

	keys[0] = kTestAuthToolCheckUIDsResponseRUID;
	keys[1] = kTestAuthToolCheckUIDsResponseEUID;
	keys[2] = kTestAuthToolCheckUIDsResponseSUID;
	
	tmp = euid;
	values[0] = CFNumberCreate(NULL, kCFNumberLongLongType, &tmp);
	tmp = ruid;
	values[1] = CFNumberCreate(NULL, kCFNumberLongLongType, &tmp);
	tmp = suid;
	values[2] = CFNumberCreate(NULL, kCFNumberLongLongType, &tmp);
	
	err = noErr;
	if (values[0] == NULL || values[1] == NULL || values[2] == NULL) {
		err = coreFoundationUnknownErr;
	}
	if (err == noErr) {
		*result = CFDictionaryCreate(NULL, (const void **) &keys, (const void **) &values, 3, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
		err = CFQError(*result);
	}
	
	CFQRelease(values[0]);
	CFQRelease(values[1]);
	CFQRelease(values[2]);

	assert( (err == noErr) == (*result != NULL) );
		
	return err;
}

static OSStatus DoAbortCommand(CFDictionaryRef *result, int abortPID)
{
	OSStatus err;
	int mib[4] = {CTL_KERN, KERN_PROC, KERN_PROC_ALL, 0};
	size_t buf_size;
	
	// elevate privs
	err = MoreSecSetPrivilegedEUID();
	
	err = sysctl(mib, 4, NULL, &buf_size, NULL, 0);
	err = MoreUNIXErrno(err);
	if (err == noErr) {
		struct kinfo_proc *processes = NULL;
    	int i,nb_entries;
    	nb_entries = buf_size / sizeof(struct kinfo_proc);
    	processes = (struct kinfo_proc*) NewPtrClear((CFIndex)buf_size);
    	if (processes != NULL) {
    		err = sysctl(mib, 4, processes, &buf_size, NULL, 0);
			err = MoreUNIXErrno(err);
    		if (err == noErr) {
			    for (i = 0; i < nb_entries; i++) {
			    	if (processes[i].kp_eproc.e_ppid != abortPID) continue;
					(void) kill(processes[i].kp_proc.p_pid, SIGTERM);
					(void) kill(processes[i].kp_proc.p_pid, SIGKILL);
			    }
    		}
    		DisposePtr((Ptr)processes);
    	}
	}
	(void) kill(abortPID, SIGTERM);
	(void) kill(abortPID, SIGKILL);

	// back down privs
	MoreSecTemporarilySetNonPrivilegedEUID();

	return err;
}

static OSStatus CommandProc(AuthorizationRef auth, CFDictionaryRef request, CFDictionaryRef *result)
{
	OSStatus err;
	CFStringRef command;
	
	assert(auth != NULL);
	assert(request != NULL);
	assert( result != NULL);
	assert(*result == NULL);
	assert(geteuid() == getuid());
	
	err = noErr;
	command = (CFStringRef) CFDictionaryGetValue(request, kTestAuthToolCommandNameKey);
	if (   (command == NULL) 
		|| (CFGetTypeID(command) != CFStringGetTypeID()) ) {
		err = paramErr;
	}
	if (err == noErr) {
		if (CFEqual(command, kTestAuthToolCheckUIDsCommand)) {

			if (err == noErr) {
				err = DoCheckUIDsCommand(result);
			}
		} else if (CFEqual(command, kTestAuthToolTestCommand)) {
			
			CFStringRef reqStr = (CFStringRef) CFDictionaryGetValue(request, kTestAuthToolTestRequestStringKey);
			err = CFQError(reqStr);
			if (err == noErr) {
				err = DoTestCommand(result, reqStr, CFDictionaryContainsKey(request, kMoreSecAsyncIdentifierKey));
			}
		} else if (CFEqual(command,kTestAuthToolActivateCommand)) {
			// do nothing
		} else if (CFEqual(command,kTestAuthToolAbortCommand)) {
			CFNumberRef processID = (CFNumberRef) CFDictionaryGetValue(request, kTestAuthToolAbortRequestProcessKey);
			err = CFQError(processID);
			if (err == noErr) {
				int abortPID;
				CFNumberGetValue(processID, kCFNumberIntType, &abortPID);
				err = DoAbortCommand(result, abortPID);
			}
		} else {
			err = paramErr;
		}
	}
	return err;
}

int main(int argc, const char *argv[])
{	
	int err;
	int result;
	AuthorizationRef auth;

	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

	auth = MoreSecHelperToolCopyAuthRef();
	
	err = MoreSecDestroyInheritedEnvironment(kMoreSecKeepStandardFilesMask, argv);
	
	if (err == 0) {
		err = MoreUNIXIgnoreSIGPIPE();
	}
	
	if (err == 0) {
		err = MoreSecHelperToolMain(STDIN_FILENO, STDOUT_FILENO, auth, CommandProc, argc, argv);
	}

	result = MoreSecErrorToHelperToolResult(err);

	[pool release];

	return result;
}
